<?php
session_start();
require_once 'db_connection.php'; // mysqli ($conn) with utf8mb4

// AuthZ
if (!isset($_SESSION['user_id']) || ($_SESSION['role'] ?? '') !== 'branch_head') {
    header("Location: login.php");
    exit();
}

$branch_id = (int)($_SESSION['branch_id'] ?? 0);
if ($branch_id <= 0) {
    header("Location: branch_head_dashboard.php");
    exit();
}

/* CSRF */
if (empty($_SESSION['csrf'])) { $_SESSION['csrf'] = bin2hex(random_bytes(32)); }
$csrf = $_SESSION['csrf'];
function check_csrf($t){ return hash_equals($_SESSION['csrf'] ?? '', $t); }

$success_message = '';
$error_message   = '';

/* Handle assignment */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_forward_to'])) {
    if (!check_csrf($_POST['csrf'] ?? '')) {
        $error_message = "Security check failed. Please retry.";
    } else {
        $letter_id  = (int)($_POST['letter_id'] ?? 0);
        $forward_to = (int)($_POST['forward_to'] ?? 0);

        if ($letter_id <= 0 || $forward_to <= 0) {
            $error_message = "Please choose a valid letter and staff member.";
        } else {
            // Assign and set status in one query; only from 'received'
            $sql = "UPDATE letters
                    SET forward_to = ?, status_branch_head = 'received_assigned', assign_date = NOW()
                    WHERE letter_id = ? AND branch_id = ? AND status_branch_head = 'received'";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("iii", $forward_to, $letter_id, $branch_id);
            if ($stmt->execute() && $stmt->affected_rows > 0) {
                $success_message = "Letter assigned to staff successfully.";
            } else {
                $error_message = "Nothing changed. The letter may already be assigned or not in 'received' state.";
            }
            $stmt->close();
        }
    }
}

/* Fetch letters currently in 'received' for this branch */
$sql = "SELECT letter_id, letter_number, letter_date, department, subject,
               received_date, scan_copy, forward_to
        FROM letters
        WHERE branch_id = ? AND status_branch_head = 'received'
        ORDER BY letter_date DESC, letter_id DESC";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $branch_id);
$stmt->execute();
$letters = $stmt->get_result();
$stmt->close();

/* Fetch active staff (not branch_head) for this branch */
$staff = [];
$usql = "SELECT user_id, username FROM users
         WHERE branch_id = ? AND role <> 'branch_head' AND status='active'
         ORDER BY username ASC";
$ustmt = $conn->prepare($usql);
$ustmt->bind_param("i", $branch_id);
$ustmt->execute();
$ures = $ustmt->get_result();
while ($u = $ures->fetch_assoc()) { $staff[] = $u; }
$ustmt->close();

// $conn->close(); // keep open if you extend below
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Received Letters — Assign to Staff</title>

  <!-- Bootstrap 5 -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css"
        rel="stylesheet"
        integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YctnYmDr5pNlyT2bRjXh0JMhjY6hW+ALEwIH"
        crossorigin="anonymous" />
  <link rel="stylesheet"
        href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css" />

  <style>
    :root { --brand:#0d6efd; --bg:#f6f8fb; }
    body { background: var(--bg); }
    .page-head {
      background: linear-gradient(135deg, var(--brand), #5aa2ff);
      color: #fff; border-radius: 1rem; padding: 1rem 1.25rem;
      box-shadow: 0 6px 24px rgba(13,110,253,.25);
    }
    .table thead th { background:#fff; position: sticky; top: 0; z-index: 2; }
    .actions { white-space: nowrap; }
  </style>
</head>
<body>
<div class="container-fluid py-3">
  <div class="page-head mb-3">
    <div class="d-flex align-items-center justify-content-between">
      <div class="d-flex align-items-center gap-3">
        <i class="fa-solid fa-box-open fs-3"></i>
        <div>
          <h5 class="mb-0 fw-bold">Received Letters (Not Assigned)</h5>
          <small>Assign to staff to continue the workflow</small>
        </div>
      </div>
      <div>
        <a href="branch_head_dashboard.php" class="btn btn-light">
          <i class="fa-solid fa-arrow-left-long me-1"></i> Back to Dashboard
        </a>
      </div>
    </div>
  </div>

  <?php if (!empty($success_message)): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
      <?= htmlspecialchars($success_message, ENT_QUOTES, 'UTF-8'); ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
  <?php endif; ?>
  <?php if (!empty($error_message)): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
      <?= htmlspecialchars($error_message, ENT_QUOTES, 'UTF-8'); ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
  <?php endif; ?>

  <?php if ($letters->num_rows === 0): ?>
    <div class="card border-0 shadow-sm">
      <div class="card-body text-center py-5">
        <i class="fa-regular fa-circle-check fa-2x mb-2 text-success"></i>
        <p class="mb-0">No received letters pending assignment.</p>
      </div>
    </div>
  <?php else: ?>
    <div class="card border-0 shadow-sm">
      <div class="table-responsive">
        <table class="table align-middle mb-0">
          <thead class="table-light">
            <tr>
              <th style="min-width:140px;">Letter No.</th>
              <th style="min-width:120px;">Date</th>
              <th style="min-width:140px;">Department</th>
              <th>Subject</th>
              <th style="min-width:130px;">Received Date</th>
              <th style="min-width:110px;">Scan Copy</th>
              <th class="text-end" style="min-width:260px;">Assign to Staff</th>
            </tr>
          </thead>
          <tbody>
          <?php while ($row = $letters->fetch_assoc()): ?>
            <tr>
              <td><?= htmlspecialchars($row['letter_number'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['letter_date'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['department'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['subject'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= !empty($row['received_date']) ? htmlspecialchars($row['received_date'], ENT_QUOTES, 'UTF-8') : '—'; ?></td>
              <td>
                <?php if (!empty($row['scan_copy'])): ?>
                  <a href="<?= htmlspecialchars($row['scan_copy'], ENT_QUOTES, 'UTF-8'); ?>" target="_blank"
                     class="btn btn-sm btn-outline-primary">
                    <i class="fa-solid fa-up-right-from-square me-1"></i> View
                  </a>
                <?php else: ?>
                  <span class="text-muted">None</span>
                <?php endif; ?>
              </td>
              <td class="text-end actions">
                <form method="post" class="row g-2 justify-content-end">
                  <input type="hidden" name="csrf" value="<?= $csrf; ?>">
                  <input type="hidden" name="letter_id" value="<?= (int)$row['letter_id']; ?>">
                  <div class="col-12 col-md-7">
                    <select name="forward_to" class="form-select form-select-sm" required>
                      <option value="">— Select Staff —</option>
                      <?php foreach ($staff as $u): ?>
                        <option value="<?= (int)$u['user_id']; ?>" <?= ((int)$row['forward_to'] === (int)$u['user_id']) ? 'selected' : ''; ?>>
                          <?= htmlspecialchars($u['username'], ENT_QUOTES, 'UTF-8'); ?>
                        </option>
                      <?php endforeach; ?>
                    </select>
                  </div>
                  <div class="col-12 col-md-5">
                    <button type="submit" name="update_forward_to" class="btn btn-primary btn-sm w-100">
                      <i class="fa-solid fa-user-check me-1"></i> Assign
                    </button>
                  </div>
                </form>
              </td>
            </tr>
          <?php endwhile; ?>
          </tbody>
        </table>
      </div>
    </div>
  <?php endif; ?>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"
        integrity="sha384-YvpcrYf0tY3lHB60NNkmXc5s9fDVZLESaAA55NDzOxhy9GkcIdslK1eN7N6jIeHz"
        crossorigin="anonymous"></script>
</body>
</html>
