<?php
session_start();

// ---- Auth: only letter_admin or super_admin ---------------------------------
if (!isset($_SESSION['role']) || !in_array($_SESSION['role'], ['letter_admin','super_admin'])) {
  header('Location: unauthorized.php');
  exit();
}

require_once 'db_connection.php'; // $conn (mysqli)

// ---- CSRF (simple) -----------------------------------------------------------
if (empty($_SESSION['csrf'])) { $_SESSION['csrf'] = bin2hex(random_bytes(16)); }
$csrf = $_SESSION['csrf'];

// ---- Filters (persist from POST) --------------------------------------------
$start_date = trim($_POST['start_date'] ?? '');
$end_date   = trim($_POST['end_date'] ?? '');
$branch_id  = trim($_POST['branch_id'] ?? '');
$priority   = trim($_POST['priority']  ?? '');

// ---- Build parameterized query ----------------------------------------------
$sql = "SELECT l.letter_id, l.letter_number, l.letter_date, l.department, l.subject,
               b.branch_name, l.scan_copy, u.username, l.assign_date, l.status_branch_head,
               l.hod_comment, l.priority
        FROM letters l
        JOIN branches b ON l.branch_id = b.branch_id
        JOIN users u    ON l.user_id   = u.user_id
        WHERE l.status_branch_head = 'pending'";

$params = [];
$types  = '';

if ($start_date !== '' && $end_date !== '') {
  $sql    .= " AND l.letter_date BETWEEN ? AND ?";
  $params[] = $start_date; $types .= 's';
  $params[] = $end_date;   $types .= 's';
}
if ($branch_id !== '') {
  $sql    .= " AND l.branch_id = ?";
  $params[] = (int)$branch_id; $types .= 'i';
}
if ($priority !== '') {
  $sql    .= " AND l.priority = ?";
  $params[] = $priority; $types .= 's';
}

$sql .= " ORDER BY l.letter_date DESC, l.letter_id DESC";

$stmt = $conn->prepare($sql);
if ($types !== '') { $stmt->bind_param($types, ...$params); }
$stmt->execute();
$result = $stmt->get_result();
$rows   = $result ? $result->fetch_all(MYSQLI_ASSOC) : [];
$stmt->close();

// Branch list for filter
$branches = [];
$rb = $conn->query("SELECT branch_id, branch_name FROM branches ORDER BY branch_name ASC");
while ($rb && $r = $rb->fetch_assoc()) { $branches[] = $r; }
$rb?->close();

$conn->close();

// Role-based back
$role = $_SESSION['role'];
$back = $role==='super_admin' ? 'super_admin_dashboard.php' : ($role==='letter_admin' ? 'letter_admin_dashboard.php' : 'branch_head_dashboard.php');
?>
<!doctype html>
<html lang="en" data-bs-theme="light">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1, viewport-fit=cover">
  <title>Pending Letters – Admin</title>
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700;800&display=swap" rel="stylesheet">

  <!-- Bootstrap 5 -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <!-- DataTables 1.13 (Bootstrap 5 theme + Buttons) -->
  <link rel="stylesheet" href="https://cdn.datatables.net/1.13.8/css/dataTables.bootstrap5.min.css">
  <link rel="stylesheet" href="https://cdn.datatables.net/buttons/3.0.2/css/buttons.bootstrap5.min.css">
  <!-- Icons -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css"/>

  <style>
    :root{ --brand:#2563eb; --brand-2:#60a5fa; --ink:#0b1320; --bg:#f5f7fb; --surface:#fff; --radius:16px; --shadow:0 10px 30px rgba(2,8,23,.07) }
    *{ box-sizing:border-box }
    body{ font-family:'Inter', system-ui, -apple-system, Segoe UI, Roboto, 'Helvetica Neue', Arial; background:var(--bg); color:var(--ink) }

    /* Header */
    .hero{ background: linear-gradient(135deg, var(--brand), var(--brand-2)); color:#fff; border-radius: 0 0 24px 24px; padding: 18px 0; box-shadow: 0 14px 34px rgba(37,99,235,.25) }
    .hero .subtitle{ opacity:.9 }

    /* Card */
    .cardx{ background:var(--surface); border:1px solid rgba(2,8,23,.06); border-radius:var(--radius); box-shadow:var(--shadow) }
    .cardx-header{ padding:14px 16px; border-bottom:1px solid rgba(2,8,23,.06); display:flex; align-items:center; justify-content:space-between }
    .cardx-body{ padding:16px }

    /* Filters */
    .filter-grid{ display:grid; grid-template-columns: repeat(4, minmax(0,1fr)); gap:12px }
    @media (max-width: 992px){ .filter-grid{ grid-template-columns: repeat(2, minmax(0,1fr)) } }
    @media (max-width: 576px){ .filter-grid{ grid-template-columns: 1fr } }

    /* Table */
    .sticky-head thead th{ position:sticky; top:0; z-index:2; background:#f8fafc }
    .priority-badge{ font-weight:700 }
    tr.high-priority{ box-shadow: inset 0 0 0 9999px rgba(239,68,68,.08) }

    /* Footer bar */
    .footerbar{ position: sticky; bottom:0; background:linear-gradient(180deg,#fff,#f7f9fc); border-top:1px solid rgba(2,8,23,.06); padding:10px 16px; display:flex; gap:12px; align-items:center; justify-content:space-between; border-radius: 14px 14px 0 0; box-shadow: 0 -8px 24px rgba(2,8,23,.06); }

    /* Dark */
    @media (prefers-color-scheme: dark){ :root{ --bg:#0b1220; --ink:#e5e7eb; --surface:#0f1629 } }
    [data-bs-theme="dark"] .hero{ box-shadow: none }
    [data-bs-theme="dark"] .cardx{ background:#0f1629; border-color: rgba(255,255,255,.08) }
    [data-bs-theme="dark"] .sticky-head thead th{ background:#0f1629 }
  </style>
</head>
<body>

  <!-- Top Hero -->
  <div class="hero mb-3">
    <div class="container-xxl d-flex align-items-center justify-content-between">
      <div class="py-2">
        <h1 class="h4 mb-1"><i class="fa-solid fa-envelope-circle-check me-2"></i>Pending Letters – Not Received by Branch Heads</h1>
        <div class="subtitle small">Review, filter and export letters with status <span class="fw-bold">pending</span>.</div>
      </div>
      <div class="d-flex align-items-center gap-2">
        <a href="<?= htmlspecialchars($back,ENT_QUOTES,'UTF-8'); ?>" class="btn btn-outline-light btn-sm"><i class="fa-solid fa-arrow-left-long me-1"></i>Dashboard</a>
        <button id="themeToggle" class="btn btn-outline-light btn-sm" title="Toggle dark mode"><i class="fa-solid fa-moon"></i></button>
      </div>
    </div>
  </div>

  <div class="container-xxl mb-4">
    <!-- Filters -->
    <div class="cardx mb-3">
      <div class="cardx-header">
        <strong><i class="fa-solid fa-filter me-2"></i>Filters</strong>
        <div class="d-flex gap-2">
          <a href="pending_with_branch_head.php" class="btn btn-outline-secondary btn-sm"><i class="fa-solid fa-rotate"></i> Reset</a>
        </div>
      </div>
      <div class="cardx-body">
        <form action="pending_with_branch_head.php" method="POST" class="filter-grid">
          <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf,ENT_QUOTES,'UTF-8'); ?>">

          <div>
            <label class="form-label small mb-1">Start Date</label>
            <input type="date" name="start_date" class="form-control form-control-sm" value="<?= htmlspecialchars($start_date,ENT_QUOTES,'UTF-8'); ?>">
          </div>
          <div>
            <label class="form-label small mb-1">End Date</label>
            <input type="date" name="end_date" class="form-control form-control-sm" value="<?= htmlspecialchars($end_date,ENT_QUOTES,'UTF-8'); ?>">
          </div>
          <div>
            <label class="form-label small mb-1">Branch</label>
            <select name="branch_id" class="form-select form-select-sm">
              <option value="">All branches</option>
              <?php foreach($branches as $b): ?>
                <option value="<?= (int)$b['branch_id']; ?>" <?= ($branch_id!=='' && (int)$branch_id===(int)$b['branch_id'])?'selected':''; ?>>
                  <?= htmlspecialchars($b['branch_name'],ENT_QUOTES,'UTF-8'); ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>
          <div>
            <label class="form-label small mb-1">Priority</label>
            <select name="priority" class="form-select form-select-sm">
              <?php $opts=[''=>'All priorities','Low'=>'Low','Medium'=>'Medium','High'=>'High']; foreach($opts as $val=>$label): ?>
                <option value="<?= htmlspecialchars($val,ENT_QUOTES,'UTF-8'); ?>" <?= ($priority===$val)?'selected':''; ?>><?= htmlspecialchars($label,ENT_QUOTES,'UTF-8'); ?></option>
              <?php endforeach; ?>
            </select>
          </div>

          <div class="d-flex align-items-end">
            <button type="submit" class="btn btn-primary btn-sm w-100"><i class="fa-solid fa-filter me-1"></i>Apply</button>
          </div>
        </form>
      </div>
    </div>

    <!-- Table Card -->
    <div class="cardx">
      <div class="cardx-header">
        <strong><i class="fa-solid fa-table me-2"></i>Results (<?= count($rows); ?>)</strong>
        <div id="dtButtons" class="d-flex flex-wrap gap-2"></div>
      </div>
      <div class="cardx-body">
        <div class="table-responsive">
          <table id="lettersTable" class="table table-hover align-middle sticky-head" style="width:100%">
            <thead class="table-light">
              <tr>
                <th>Letter ID</th>
                <th>Letter Number</th>
                <th>Letter Date</th>
                <th>Department</th>
                <th>Subject</th>
                <th>Branch</th>
                <th>Scan</th>
                <th>User</th>
                <th>Assign Date</th>
                <th>Status</th>
                <th>HOD Comment</th>
                <th>Priority</th>
              </tr>
            </thead>
            <tbody>
              <?php if (!empty($rows)): foreach($rows as $row):
                $isHigh = (strtolower($row['priority']??'')==='high');
                $cls = $isHigh ? 'high-priority' : '';
                $scan = htmlspecialchars($row['scan_copy'] ?? '', ENT_QUOTES, 'UTF-8');

                // Decide link: if URL, link; if file exists on server path, link; else text
                $scanHtml = 'No Scan';
                if ($scan) {
                  if (filter_var($scan, FILTER_VALIDATE_URL)) {
                    $scanHtml = "<a href='{$scan}' target='_blank' rel='noopener'>View</a>";
                  } elseif (file_exists($scan)) {
                    $href = htmlspecialchars($scan,ENT_QUOTES,'UTF-8');
                    $scanHtml = "<a href='{$href}' target='_blank' rel='noopener'>View</a>";
                  }
                }

                $priorityBadge = '<span class="badge text-bg-secondary priority-badge">N/A</span>';
                $p = strtolower($row['priority'] ?? '');
                if ($p==='low')    $priorityBadge = '<span class="badge text-bg-success priority-badge">Low</span>';
                if ($p==='medium') $priorityBadge = '<span class="badge text-bg-warning priority-badge">Medium</span>';
                if ($p==='high')   $priorityBadge = '<span class="badge text-bg-danger priority-badge">High</span>';
              ?>
              <tr class="<?= $cls; ?>">
                <td><?= (int)$row['letter_id']; ?></td>
                <td><?= htmlspecialchars($row['letter_number'] ?? '', ENT_QUOTES, 'UTF-8'); ?></td>
                <td><?= htmlspecialchars($row['letter_date'] ?? '', ENT_QUOTES, 'UTF-8'); ?></td>
                <td><?= htmlspecialchars($row['department'] ?? '', ENT_QUOTES, 'UTF-8'); ?></td>
                <td><?= htmlspecialchars($row['subject'] ?? '', ENT_QUOTES, 'UTF-8'); ?></td>
                <td><?= htmlspecialchars($row['branch_name'] ?? '', ENT_QUOTES, 'UTF-8'); ?></td>
                <td><?= $scanHtml; ?></td>
                <td><?= htmlspecialchars($row['username'] ?? '', ENT_QUOTES, 'UTF-8'); ?></td>
                <td><?= htmlspecialchars($row['assign_date'] ?? '', ENT_QUOTES, 'UTF-8'); ?></td>
                <td><span class="badge text-bg-info"><?= htmlspecialchars($row['status_branch_head'] ?? '', ENT_QUOTES, 'UTF-8'); ?></span></td>
                <td><?= htmlspecialchars($row['hod_comment'] ?? '', ENT_QUOTES, 'UTF-8'); ?></td>
                <td><?= $priorityBadge; ?></td>
              </tr>
              <?php endforeach; endif; ?>
            </tbody>
          </table>
        </div>
      </div>
    </div>

    <!-- Footer actions -->
    <div class="footerbar mt-3">
      <div class="d-flex align-items-center gap-2">
        <a href="<?= htmlspecialchars($back,ENT_QUOTES,'UTF-8'); ?>" class="btn btn-outline-secondary btn-sm"><i class="fa-solid fa-arrow-left-long me-1"></i>Back</a>
      </div>
      <div class="d-flex align-items-center gap-2">
        <button class="btn btn-primary btn-sm" onclick="window.print()"><i class="fa-solid fa-print me-1"></i>Print</button>
        <button id="scrollTop" class="btn btn-outline-primary btn-sm" title="Scroll to top"><i class="fa-solid fa-arrow-up"></i></button>
      </div>
    </div>
  </div>

  <!-- JS -->
  <script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
  <script src="https://cdn.datatables.net/1.13.8/js/jquery.dataTables.min.js"></script>
  <script src="https://cdn.datatables.net/1.13.8/js/dataTables.bootstrap5.min.js"></script>
  <script src="https://cdn.datatables.net/buttons/3.0.2/js/dataTables.buttons.min.js"></script>
  <script src="https://cdn.datatables.net/buttons/3.0.2/js/buttons.bootstrap5.min.js"></script>
  <script src="https://cdnjs.cloudflare.com/ajax/libs/jszip/3.10.1/jszip.min.js"></script>
  <script src="https://cdnjs.cloudflare.com/ajax/libs/pdfmake/0.2.7/pdfmake.min.js"></script>
  <script src="https://cdnjs.cloudflare.com/ajax/libs/pdfmake/0.2.7/vfs_fonts.js"></script>
  <script src="https://cdn.datatables.net/buttons/3.0.2/js/buttons.html5.min.js"></script>
  <script src="https://cdn.datatables.net/buttons/3.0.2/js/buttons.print.min.js"></script>
  <script src="https://cdn.datatables.net/buttons/3.0.2/js/buttons.colVis.min.js"></script>

  <script>
    // Theme toggle (persist)
    (function(){
      const btn = document.getElementById('themeToggle');
      const root = document.documentElement;
      const saved = localStorage.getItem('theme');
      if (saved) {
        root.setAttribute('data-bs-theme', saved);
      } else if (matchMedia('(prefers-color-scheme: dark)').matches) {
        root.setAttribute('data-bs-theme', 'dark');
      }
      btn?.addEventListener('click', ()=>{
        const cur = root.getAttribute('data-bs-theme') === 'dark' ? 'light' : 'dark';
        root.setAttribute('data-bs-theme', cur);
        localStorage.setItem('theme', cur);
      });
    })();

    // DataTable + buttons
    $(function(){
      const title = 'Pending Letters (' + new Date().toISOString().slice(0,10) + ')';
      const dt = $('#lettersTable').DataTable({
        pageLength: 25,
        stateSave: true,
        order: [[2,'desc']], // letter_date
        autoWidth: false,
        dom: 't',
        columnDefs:[ {targets:[0,2,8], width:'90px'}, {targets:[6,10,11], orderable:false} ],
        language: { emptyTable: 'No pending letters found.' },
        deferRender: true
      });

      new $.fn.dataTable.Buttons(dt, {
        buttons: [
          { extend:'colvis', className:'btn btn-outline-primary btn-sm', text:'Columns' },
          { extend:'copyHtml5',  className:'btn btn-outline-primary btn-sm', title },
          { extend:'csvHtml5',   className:'btn btn-outline-primary btn-sm', title, filename:'pending_letters' },
          { extend:'excelHtml5', className:'btn btn-outline-primary btn-sm', title, filename:'pending_letters' },
          { extend:'pdfHtml5',   className:'btn btn-outline-primary btn-sm', title, orientation:'landscape', pageSize:'A4' },
          { extend:'print',      className:'btn btn-outline-primary btn-sm', title }
        ]
      });
      dt.buttons().container().appendTo('#dtButtons');

      // Scroll top
      document.getElementById('scrollTop').addEventListener('click', ()=>window.scrollTo({top:0, behavior:'smooth'}));
    });
  </script>
</body>
</html>
