<?php
// Include database connection
include '..\db_connection.php';

// Start session and check if the user is logged in
session_start();
if (!isset($_SESSION['user_id'])) {
    echo "<p>You need to be logged in to view the letters.</p>";
    exit;
}

// Get the current user id from the session
$user_id = $_SESSION['user_id'];
$user_role = $_SESSION['role']; // Assuming role is stored in the session

// Query based on role
if ($user_role == 'staff_user') {
    // Staff user should view their drafts with status 'approved'
    $query = "SELECT draft_id, draft_number, subject, department, status, attachment 
              FROM letter_drafts 
              WHERE staff_user_id = ? AND status = 'drafted'";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("i", $user_id);  // Staff user ID
} else if ($user_role == 'branch_head' || $user_role == 'super_admin') {
    // Approvers should see drafts assigned to them with 'approved' status
    $query = "SELECT draft_id, draft_number, subject, department, status, attachment 
              FROM letter_drafts 
              WHERE approver_id = ? AND status = 'drafted'";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("i", $user_id);  // Approver ID
}

// Execute the query
$stmt->execute();
$result = $stmt->get_result();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>View Approved Letters</title>
    <link rel="stylesheet" href="https://maxcdn.bootstrapcdn.com/bootstrap/4.0.0/css/bootstrap.min.css">
</head>
<body>
    <div class="container">
        <h2 class="mt-4">View Approved Letters</h2>

        <!-- Display message if set -->
        <?php if (isset($_SESSION['message'])): ?>
            <div class="alert alert-info"><?php echo $_SESSION['message']; ?></div>
            <?php unset($_SESSION['message']); // Clear the message after displaying it ?>
        <?php endif; ?>

        <?php if ($result->num_rows > 0): ?>
            <table class="table">
                <thead>
                    <tr>
                        <th>Draft Number</th>
                        <th>Subject</th>
                        <th>Department</th>
                        <th>Status</th>
                        <th>Attachment</th> <!-- New Column for Attachment -->
                    </tr>
                </thead>
                <tbody>
                    <?php while ($row = $result->fetch_assoc()): ?>
                        <tr>
                            <td><?php echo $row['draft_number']; ?></td>
                            <td><?php echo $row['subject']; ?></td>
                            <td><?php echo $row['department']; ?></td>
                            <td><?php echo $row['status']; ?></td>
                            <td>
                                <?php if ($row['attachment']): ?>
                                    <!-- Display the download button if there is an attachment -->
                                    <a href="<?php echo $row['attachment']; ?>" class="btn btn-info btn-sm" download>Download Attachment</a>
                                <?php else: ?>
                                    No attachment
                                <?php endif; ?>
                            </td>
                            <td>
                                <!-- Action buttons for approving or rejecting the draft -->
                                <button class="btn btn-success btn-sm" data-toggle="modal" data-target="#approveModal<?php echo $row['draft_id']; ?>">Approve</button>
                                <button class="btn btn-danger btn-sm" data-toggle="modal" data-target="#rejectModal<?php echo $row['draft_id']; ?>">Reject</button>
                            </td>
                        </tr>

                        <!-- Approve Modal -->
                        <div class="modal fade" id="approveModal<?php echo $row['draft_id']; ?>" tabindex="-1" role="dialog" aria-labelledby="approveModalLabel<?php echo $row['draft_id']; ?>" aria-hidden="true">
                            <div class="modal-dialog" role="document">
                                <div class="modal-content">
                                    <div class="modal-header">
                                        <h5 class="modal-title" id="approveModalLabel<?php echo $row['draft_id']; ?>">Approve Letter</h5>
                                        <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                                            <span aria-hidden="true">&times;</span>
                                        </button>
                                    </div>
                                    <form action="process_approval.php" method="POST" enctype="multipart/form-data">
                                        <div class="modal-body">
                                            <div class="form-group">
                                                <label for="comment">Approval Comments</label>
                                                <textarea name="comment" class="form-control" id="comment" rows="3" required></textarea>
                                            </div>
                                            
                                            <!-- Display current attachment and provide an option to upload a new file -->
                                            <div class="form-group">
                                                <label for="new_attachment">New Attachment (optional)</label>
                                                <input type="file" name="new_attachment" class="form-control" accept=".pdf, .docx, .doc, .jpg, .png">
                                            </div>
                                            
                                            <input type="hidden" name="draft_id" value="<?php echo $row['draft_id']; ?>">
                                            <input type="hidden" name="approval" value="approved">
                                        </div>
                                        <div class="modal-footer">
                                            <button type="button" class="btn btn-secondary" data-dismiss="modal">Close</button>
                                            <button type="submit" class="btn btn-success">Approve</button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>

                        <!-- Reject Modal -->
                        <div class="modal fade" id="rejectModal<?php echo $row['draft_id']; ?>" tabindex="-1" role="dialog" aria-labelledby="rejectModalLabel<?php echo $row['draft_id']; ?>" aria-hidden="true">
                            <div class="modal-dialog" role="document">
                                <div class="modal-content">
                                    <div class="modal-header">
                                        <h5 class="modal-title" id="rejectModalLabel<?php echo $row['draft_id']; ?>">Reject Letter</h5>
                                        <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                                            <span aria-hidden="true">&times;</span>
                                        </button>
                                    </div>
                                    <form action="process_approval.php" method="POST">
                                        <div class="modal-body">
                                            <div class="form-group">
                                                <label for="comment">Rejection Comments</label>
                                                <textarea name="comment" class="form-control" id="comment" rows="3" required></textarea>
                                            </div>
                                            <input type="hidden" name="draft_id" value="<?php echo $row['draft_id']; ?>">
                                            <input type="hidden" name="approval" value="rejected">
                                        </div>
                                        <div class="modal-footer">
                                            <button type="button" class="btn btn-secondary" data-dismiss="modal">Close</button>
                                            <button type="submit" class="btn btn-danger">Reject</button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                    <?php endwhile; ?>
                </tbody>
            </table>
        <?php else: ?>
            <p>No approved letters found for you.</p>
        <?php endif; ?>
    </div>

    <!-- Bootstrap JS and dependencies -->
    <script src="https://code.jquery.com/jquery-3.2.1.slim.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/popper.js/1.12.9/umd/popper.min.js"></script>
    <script src="https://maxcdn.bootstrapcdn.com/bootstrap/4.0.0/js/bootstrap.min.js"></script>
</body>
</html>
