<?php
// get_user_letters.php (Bootstrap 4-compatible + username fallback)
declare(strict_types=1);

session_start();
require_once 'db_connection.php'; // must set $conn (mysqli) + utf8mb4

header('Content-Type: text/html; charset=UTF-8');

// --- Read inputs ---
$user_id   = isset($_GET['user_id'])   ? (int)$_GET['user_id']   : 0;
$branch_id = isset($_GET['branch_id']) ? (int)$_GET['branch_id'] : 0;
$username  = isset($_GET['username'])  ? trim((string)$_GET['username']) : '';

// --- Optional fallback: resolve username -> user_id if user_id missing ---
if ($user_id <= 0 && $username !== '') {
    if ($find = $conn->prepare("SELECT user_id FROM users WHERE username = ? LIMIT 1")) {
        $find->bind_param('s', $username);
        $find->execute();
        $find->bind_result($uid);
        if ($find->fetch()) { $user_id = (int)$uid; }
        $find->close();
    }
}

if ($user_id <= 0) {
    http_response_code(400);
    echo '<div class="alert alert-warning mb-0">Invalid user selection.</div>';
    exit;
}

// --- Build SQL (branch_id optional) ---
$conds  = ['l.user_id = ?'];
$params = [$user_id];
$types  = 'i';

if ($branch_id > 0) {
    $conds[] = 'l.branch_id = ?';
    $params[] = $branch_id;
    $types   .= 'i';
}

$where = 'WHERE '.implode(' AND ', $conds);

$sql = "
SELECT
  l.letter_id,
  l.letter_number,
  l.letter_date,
  l.department,
  l.subject,
  l.scan_copy,
  l.seen,
  l.action,
  l.priority,
  l.status_branch_head,
  u.username AS forward_to_user,
  b.branch_name
FROM letters l
LEFT JOIN users u    ON l.forward_to = u.user_id
LEFT JOIN branches b ON l.branch_id  = b.branch_id
{$where}
ORDER BY l.letter_date DESC, l.letter_id DESC
";

// --- Run query safely ---
$stmt = $conn->prepare($sql);
if (!$stmt) {
    http_response_code(500);
    echo '<div class="alert alert-danger mb-0">Query prepare failed: '.htmlspecialchars($conn->error, ENT_QUOTES, 'UTF-8').'</div>';
    exit;
}
$stmt->bind_param($types, ...$params);
$stmt->execute();
$res = $stmt->get_result();

if (!$res || $res->num_rows === 0) {
    echo '<div class="alert alert-info mb-0">No letters found for this selection.</div>';
    $stmt->close(); $conn->close();
    exit;
}

// --- Helper to map to Bootstrap 4 badge classes ---
function badgeSeen($seen): string {
    return ((int)$seen === 1)
        ? '<span class="badge badge-success">Yes</span>'
        : '<span class="badge badge-danger">No</span>';
}
function badgeAction($action): string {
    if ($action === 'yes') return '<span class="badge badge-success">Yes</span>';
    if ($action === 'no')  return '<span class="badge badge-danger">No</span>';
    return '<span class="badge badge-secondary">—</span>';
}
function badgePriority($p): string {
    $p = (string)$p;
    if ($p === 'High')   return '<span class="badge badge-danger">High</span>';
    if ($p === 'Medium') return '<span class="badge badge-warning">Medium</span>';
    if ($p === 'Low')    return '<span class="badge badge-success">Low</span>';
    return '<span class="badge badge-secondary">—</span>';
}

// --- Render table (note: id added for optional DataTables init) ---
?>
<div class="table-responsive">
  <table id="userLettersTable" class="table table-sm table-striped table-bordered align-middle">
    <thead class="thead-light">
      <tr>
        <th>Letter No.</th>
        <th>Date</th>
        <th>Department</th>
        <th>Subject</th>
        <th>Branch</th>
        <th>Forward To</th>
        <th>Seen</th>
        <th>Action</th>
        <th>Priority</th>
        <th>Scan</th>
      </tr>
    </thead>
    <tbody>
    <?php while ($r = $res->fetch_assoc()): ?>
      <tr>
        <td><?= htmlspecialchars($r['letter_number'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
        <td><?= htmlspecialchars($r['letter_date']   ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
        <td><?= htmlspecialchars($r['department']    ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
        <td><?= htmlspecialchars($r['subject']       ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
        <td><?= htmlspecialchars($r['branch_name']   ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
        <td><?= htmlspecialchars($r['forward_to_user'] ?: 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
        <td><?= badgeSeen($r['seen']); ?></td>
        <td><?= badgeAction($r['action'] ?? ''); ?></td>
        <td><?= badgePriority($r['priority'] ?? ''); ?></td>
        <td>
          <?php if (!empty($r['scan_copy'])): ?>
            <a href="<?= htmlspecialchars($r['scan_copy'], ENT_QUOTES, 'UTF-8'); ?>" target="_blank" class="btn btn-outline-primary btn-sm">View</a>
            <a href="<?= htmlspecialchars($r['scan_copy'], ENT_QUOTES, 'UTF-8'); ?>" download class="btn btn-outline-success btn-sm">Download</a>
          <?php else: ?>
            <span class="text-muted">N/A</span>
          <?php endif; ?>
        </td>
      </tr>
    <?php endwhile; ?>
    </tbody>
  </table>
</div>
<?php
$stmt->close();
$conn->close();
