<?php
session_start();
require_once 'db_connection.php';

/* ---------------- CSRF TOKEN ---------------- */
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

/* ---------------- Defaults ---------------- */
$username = $branch_id = $role = $status = '';
$errors = [];
$success_msg = '';

/* ---------------- Fetch branches ---------------- */
$branches_result = [];
if ($stmt = $conn->prepare("SELECT branch_id, branch_name FROM branches ORDER BY branch_name ASC")) {
    $stmt->execute();
    $res = $stmt->get_result();
    while ($row = $res->fetch_assoc()) {
        $branches_result[] = $row;
    }
    $stmt->close();
} else {
    $errors[] = "Unable to load branches.";
}

/* ---------------- Handle POST ---------------- */
if ($_SERVER["REQUEST_METHOD"] === "POST") {
    // CSRF
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $errors[] = "Invalid form submission. Please try again.";
    }

    // Collect + trim
    $username = trim($_POST['username'] ?? '');
    $password_raw = $_POST['password'] ?? '';
    $branch_id = trim($_POST['branch_id'] ?? '');
    $role = trim($_POST['role'] ?? '');
    $status = trim($_POST['status'] ?? '');

    // Validate
    if ($username === '' || strlen($username) < 3) {
        $errors[] = "Username must be at least 3 characters.";
    }
    if ($password_raw === '' || strlen($password_raw) < 6) {
        $errors[] = "Password must be at least 6 characters.";
    }

    // Whitelists
    $allowed_roles = ['letter_admin','branch_head','staff_user','super_admin'];
    $allowed_status = ['active','inactive'];
    if (!in_array($role, $allowed_roles, true)) {
        $errors[] = "Invalid role selected.";
    }
    if (!in_array($status, $allowed_status, true)) {
        $errors[] = "Invalid status selected.";
    }
    if ($branch_id === '') {
        $errors[] = "Please select a branch.";
    }

    // Check that branch exists
    if ($branch_id !== '') {
        if ($stmt = $conn->prepare("SELECT 1 FROM branches WHERE branch_id = ?")) {
            $stmt->bind_param("s", $branch_id);
            $stmt->execute();
            $stmt->store_result();
            if ($stmt->num_rows === 0) {
                $errors[] = "Selected branch does not exist.";
            }
            $stmt->close();
        } else {
            $errors[] = "Unable to verify branch.";
        }
    }

    // Unique username
    if (!$errors) {
        if ($stmt = $conn->prepare("SELECT 1 FROM users WHERE username = ?")) {
            $stmt->bind_param("s", $username);
            $stmt->execute();
            $stmt->store_result();
            if ($stmt->num_rows > 0) {
                $errors[] = "Username already exists. Please choose another.";
            }
            $stmt->close();
        } else {
            $errors[] = "Unable to validate username.";
        }
    }

    // Insert if clean
    if (!$errors) {
        $password_hashed = password_hash($password_raw, PASSWORD_DEFAULT);

        if ($stmt = $conn->prepare("INSERT INTO users (username, password, branch_id, role, status) VALUES (?, ?, ?, ?, ?)")) {
            $stmt->bind_param("sssss", $username, $password_hashed, $branch_id, $role, $status);
            if ($stmt->execute()) {
                $success_msg = "New user created successfully.";
                // Reset form fields (keep branch/role/status if you prefer)
                $username = '';
                // Regenerate CSRF token after successful POST to prevent resubmission
                $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
            } else {
                $errors[] = "Something went wrong while saving. Please try again.";
            }
            $stmt->close();
        } else {
            $errors[] = "Failed to prepare save statement.";
        }
    }
}

$conn->close();

/* ---------------- Helper ---------------- */
function h($str){ return htmlspecialchars($str ?? '', ENT_QUOTES, 'UTF-8'); }
?>
<!doctype html>
<html lang="en" data-bs-theme="light">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Add User · Office Management System</title>

  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet" referrerpolicy="no-referrer" />

  <style>
    body{
      min-height:100vh;
      background: linear-gradient(135deg, #0b2545 0%, #13315c 100%);
      background-attachment: fixed;
      display:flex; align-items:center; justify-content:center;
      padding: 24px 12px;
    }
    .card{
      max-width: 560px;
      width:100%;
      border:0;
      border-radius: 18px;
      box-shadow: 0 14px 34px rgba(0,0,0,.25);
      overflow:hidden;
    }
    .card-header{
      background:#ffffff;
      border-bottom:1px solid rgba(0,0,0,.06);
      padding: 1rem 1.25rem;
    }
    .card-header h1{
      font-size:1.15rem; margin:0; font-weight:700;
      display:flex; align-items:center; gap:.6rem;
    }
    .card-body{
      background:#ffffff;
      padding: 1.25rem;
    }
    .brand-top{
      display:flex; align-items:center; justify-content:space-between;
      margin-bottom:.25rem;
    }
    .brand-top img{ width:40px; height:40px; object-fit:contain; }
    .form-text{ font-size:.85rem; }
    .footer-actions{
      display:flex; gap:.5rem; flex-wrap:wrap;
    }
  </style>
</head>
<body>

<div class="card">
  <div class="card-header">
    <div class="brand-top">
      <div class="d-flex align-items-center gap-2">
        <img src="logo1.png" alt="Left Logo">
        <div>
          <strong>Office Management System</strong><br>
          <span class="text-muted">Valvettiturai Urban Council</span>
        </div>
      </div>
      <img src="logo.jpg" alt="Right Logo">
    </div>
    <h1><i class="fa-solid fa-user-plus"></i> Add New User</h1>
  </div>

  <div class="card-body">
    <?php if ($success_msg): ?>
      <div class="alert alert-success d-flex align-items-center" role="alert">
        <i class="fa-solid fa-circle-check me-2"></i>
        <div><?= h($success_msg) ?></div>
      </div>
    <?php endif; ?>

    <?php if ($errors): ?>
      <div class="alert alert-danger" role="alert">
        <strong>We couldn't save the user:</strong>
        <ul class="mb-0">
          <?php foreach ($errors as $e): ?>
            <li><?= h($e) ?></li>
          <?php endforeach; ?>
        </ul>
      </div>
    <?php endif; ?>

    <form method="POST" novalidate>
      <input type="hidden" name="csrf_token" value="<?= h($_SESSION['csrf_token']); ?>">

      <div class="mb-3">
        <label for="username" class="form-label">Username <span class="text-danger">*</span></label>
        <input type="text" id="username" name="username" class="form-control" required value="<?= h($username); ?>" autocomplete="username">
        <div class="form-text">Minimum 3 characters. Use only your official username.</div>
      </div>

      <div class="mb-3">
        <label for="password" class="form-label">Password <span class="text-danger">*</span></label>
        <div class="input-group">
          <input type="password" id="password" name="password" class="form-control" required minlength="6" autocomplete="new-password">
          <button type="button" class="btn btn-outline-secondary" id="togglePwd" aria-label="Show password">
            <i class="fa-solid fa-eye"></i>
          </button>
        </div>
        <div class="form-text">Minimum 6 characters. A mix of letters & numbers is recommended.</div>
      </div>

      <div class="mb-3">
        <label for="branch_id" class="form-label">Branch <span class="text-danger">*</span></label>
        <select id="branch_id" name="branch_id" class="form-select" required>
          <option value="">Select Branch</option>
          <?php foreach ($branches_result as $b): ?>
            <option value="<?= h($b['branch_id']); ?>" <?= ($branch_id === (string)$b['branch_id'] ? 'selected' : '') ?>>
              <?= h($b['branch_name']); ?>
            </option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="row g-3">
        <div class="col-sm-6">
          <label for="role" class="form-label">Role <span class="text-danger">*</span></label>
          <select id="role" name="role" class="form-select" required>
            <?php
              $roleOptions = ['letter_admin'=>'Letter Admin','branch_head'=>'Branch Head','staff_user'=>'Staff User','super_admin'=>'Super Admin'];
              foreach ($roleOptions as $val=>$label):
            ?>
              <option value="<?= h($val); ?>" <?= ($role===$val?'selected':'') ?>><?= h($label); ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="col-sm-6">
          <label for="status" class="form-label">Status <span class="text-danger">*</span></label>
          <select id="status" name="status" class="form-select" required>
            <option value="active" <?= ($status==='active'?'selected':'') ?>>Active</option>
            <option value="inactive" <?= ($status==='inactive'?'selected':'') ?>>Inactive</option>
          </select>
        </div>
      </div>

      <div class="mt-4 footer-actions">
        <button type="submit" class="btn btn-primary">
          <i class="fa-solid fa-floppy-disk me-1"></i> Save User
        </button>
        <a href="super_admin_dashboard.php" class="btn btn-outline-secondary">
          <i class="fa-solid fa-arrow-left-long me-1"></i> Back to Dashboard
        </a>
      </div>
    </form>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
  // Show/Hide password
  const toggle = document.getElementById('togglePwd');
  const pwd = document.getElementById('password');
  toggle?.addEventListener('click', () => {
    const isPwd = pwd.getAttribute('type') === 'password';
    pwd.setAttribute('type', isPwd ? 'text' : 'password');
    toggle.innerHTML = isPwd ? '<i class="fa-solid fa-eye-slash"></i>' : '<i class="fa-solid fa-eye"></i>';
    toggle.setAttribute('aria-label', isPwd ? 'Hide password' : 'Show password');
  });
</script>
</body>
</html>
